{ ********************************************************************************** }
{ **** Procedures for drawing - backgrounds **************************************** }
{ ****                        - Axii *********************************************** }
{ ****                        - Titles ********************************************* }
{ ****                        - Values ********************************************* }
{ ****                        - etc.  for general graphing routines **************** }
{ ***** Define your own procedures for drawing the data specific to your graph ***** }
{ **** This object was written and created by Keith Moore,                      **** }
{ ****                                        Process Automation Dept.          **** }
{ ****                                        UCB Films plc.                    **** }
{ ****                                        Wigton                            **** }
{ ********************************************************************************** }

unit Graphs;

interface

uses
  Rotate, SysUtils, WinTypes, WinProcs, Classes, Graphics, Controls, Forms, Printers;

const
  Seek_Begin = 0;                                                  { Constants for file searching }
  Seek_Current = 1;
  Seek_End = 2;
  CRLF = Chr( 13 ) + Chr( 10 );                              { Carridge Return Line Feed constant }

type
  TDoubleStream = class( TPersistent )
  private
    Data  : TMemoryStream;

    function GetSize : LongInt;                                          { Property read function }
  public
    constructor Create;
    destructor Free;

    function Read(Idx : LongInt) : Double;                         { Read a value from the stream }
    procedure Write(Idx : LongInt; Value : Double);                 { Write a value to the stream }
    procedure Add(Value : Double);                     { Add a new value to the end of the stream }
    procedure Clear;                                              { Delete all data in the stream }
    procedure SaveToFile(Filename : String);                         { Saves the stream to a file }
    procedure LoadFromFile(Filename : String);                     { Loads the stream from a file }

    property Size : LongInt read GetSize;                        { Returns the size of the stream }
  end;
  TAxis = class( TPersistent )                               { Axis object of the graph component }
  private
    { *** Property variables *** }
    FOnValueChange       : TNotifyEvent;
    FDivisions, FValues  : Byte;
    FRealMaxValue,
    FMaxValue, FMinValue : Double;
    FTitle               : String;
    FGridlines           : Boolean;

    { *** Property setting procedures *** }
    procedure SetDivisions(Value : Byte);
    procedure SetGridlines(Value : Boolean);
    procedure SetMaxValue(Value : Double);
    procedure SetMinValue(Value : Double);
    procedure SetTitle(Value : String);
    procedure SetValues(Value : Byte);

    procedure SetRealMaxValue;
  public
    property OnValueChange : TNotifyEvent read FOnValueChange
             write FOnValueChange;
    { *** Read only Property *** }
    { ** Maximum value on the axis (not always the same as MaxValue) }
    property RealMaxValue : Double read FRealMaxValue;

    { *** Control functions *** }
    function AreDivisions : Boolean;
    function Spacing : Byte;
    constructor Create;
  published
    { *** Properties *** }
    { ** Number of divisions along a particular axis ** }
    property Divisions : Byte read FDivisions write SetDivisions default 5;
    { ** Draw gridlines across the plot area (true, false) }
    property Gridlines : Boolean read FGridlines write SetGridlines default False;
    { ** Maximum value to be displayed on the axis ** }
    property MaxValue : Double read FMaxValue write SetMaxValue;
    { ** Minimum value to be displayed on the axis ** }
    property MinValue : Double read FMinValue write SetMinValue;
    { ** Axis title ** }
    property Title : String read FTitle write SetTitle;
    { ** Number of values to display at the divisions on the axis ** }
    property Values : Byte read FValues write SetValues default 3;
  end;

  TCustomGraph = class( TGraphicControl )
  private
    { **** Property Variables **** }
    FPrintLeft,
    FPrintTop,
    FPrintHeight,
    FPrintWidth          : LongInt;
    FBorderStyle,
    FPlotAreaBorderStyle : TBorderStyle;
    FAxisColor           : TColor;
    FY_Axis, FX_Axis     : TAxis;

    {**** General Procedures **** }
    procedure AxisValueChange(Sender : TObject);

    { **** Property Setting Procedures **** }
    procedure SetPrintLeft(Value : LongInt);
    procedure SetPrintTop(Value : LongInt);
    procedure SetPrintHeight(Value : LongInt);
    procedure SetPrintWidth(Value : LongInt);
    procedure SetAxisColor(Value : TColor);
    procedure SetBorderStyle(Value : TBorderStyle);
    procedure SetPlotAreaBorderStyle(Value : TBorderStyle);
    procedure SetX_Axis(Value : TAxis);
    procedure SetY_Axis(Value : TAxis);
  protected
    { **** General Varaibles **** }
    RePaintGraph         : Boolean;

    { **** Main Graph Rectange **** }
    GraphRect : TRect;

    { **** Printing Procedures **** }
    procedure PrintGraph; virtual;
      procedure PrintGridlines; virtual;
        procedure PrintGraphBorder; virtual;
        procedure PrintXGridlines; virtual;
        procedure PrintYGridlines; virtual;
      procedure SetPrinterRects(var XRect, YRect : TRect); virtual;
      procedure InitPrintImage(var PRect : TRect); virtual;
      procedure PrintXAxis(XRect : TRect); virtual;
        procedure PrintXAxisValues(XRect : TRect); virtual;
        procedure PrintXAxisDivisions(XRect : TRect); virtual;
        procedure PrintXAxisTitle(XRect : TRect); virtual;
      procedure PrintYAxis(YRect : TRect); virtual;
        procedure PrintYAxisValues(YRect : TRect); virtual;
        procedure PrintYAxisDivisions(YRect : TRect); virtual;
        procedure PrintYAxisTitle(YRect : TRect); virtual;
      procedure PrintDataLines(VP, HP : Single); virtual; abstract;

    { **** Painting Procedures **** }
    procedure InitImage(var AnImage : TBitmap; BWidth, BHeight : Integer; Col : TColor); virtual;
    procedure ImageCopy(DestC : TCanvas; DestR : TRect; SourceC : TCanvas); virtual;
    procedure PaintBackground(var AnImage : TBitmap; Col : TColor); virtual;
      procedure InitDrawGraph(var AnImage : TBitmap; var PRect : TRect); virtual;
      procedure GetRects(Sizer : TCanvas; SourceRect : TRect; var YRect, XRect, Overlap : TRect);
                virtual;
        procedure GetTextDimensions(Sizer : TCanvas; var XExtra, XOffset, YExtra,
                                    YOffset, XMax, YMax : Word); virtual;
          function GetXText(Sizer : TCanvas; var XExtra, XOffset : Word) : Word;
          function GetYText(Sizer : TCanvas; var YExtra, YOffset : Word) : Word;
        procedure GetGraphRect(Source : TRect; XExtra, YExtra, XMax, YMax : Word); virtual;
        function GetYAxisRect(Source : TRect; YOffset : Word) : TRect; virtual;
        function GetXAxisRect(Source : TRect; XOffset : Word) : TRect; virtual;
      procedure PaintXAxis(var AnImage : TBitmap; XRect : TRect); virtual;
        procedure VertLine(var AnImage : TBitmap; X, Y, Distance : Integer);
        procedure PaintXAxisValues(var AnImage : TBitmap; XRect : TRect); virtual;
        procedure PaintXAxisTitle(var AnImage : TBitmap;XRect : TRect); virtual;
        procedure PaintXAxisDivisions(var AnImage : TBitmap; XRect : TRect); virtual;
      procedure PaintYAxis(var AnImage : TBitmap; YRect, Overlap : TRect); virtual;
        procedure HorizLine(var AnImage : TBitmap; X, Y, Distance : Integer);
        procedure PaintYAxisTitle(var AnImage : TBitmap; YRect : TRect); virtual;
        procedure PaintYAxisValues(var AnImage : TBitmap; YRect : TRect); virtual;
        procedure PaintYAxisDivisions(var AnImage : TBitmap; YRect : TRect); virtual;
      procedure PaintGraph(var AnImage : TBitmap); virtual;
        procedure PaintGridlines(var AnImage : TBitmap); virtual;
          procedure PaintGraphBorder(var AnImage : TBitmap); virtual;
          procedure PaintXGridlines(var AnImage : TBitmap); virtual;
          procedure PaintYGridlines(var AnImage : TBitmap); virtual;
        procedure GetCalibration(var VP, HP, SX, EX, Step : Single); virtual; abstract;
        procedure PaintDataLines(var AnImage : TBitmap; VP, HP : Single); virtual; abstract;

    { **** Properties **** }
    property Align;
    { ** Color of axis lines ** }
    property AxisColor : TColor read FAxisColor write SetAxisColor;
    { ** Border of component (single, none) }
    property BorderStyle : TBorderStyle read FBorderStyle write SetBorderStyle;
    property Color;
    { ** Border around plot area (single, none) }
    property PlotAreaBorderStyle : TBorderStyle read FPlotAreaBorderStyle
             write SetPlotAreaBorderStyle;
    { ** Height of graph when being printed, in pixels ** }
    property PrintHeight : LongInt read FPrintHeight write SetPrintHeight;
    { ** Location of graph when being printed, in pixels ** }
    property PrintLeft : LongInt read FPrintLeft write SetPrintLeft;
    property PrintTop : LongInt read FPrintTop write SetPrintTop;
    { ** Width of graph when being printed, in pixels ** }
    property PrintWidth : LongInt read FPrintWidth write SetPrintWidth;
    { ** X axis configuration ** }
    property X_Axis : TAxis read FX_Axis write SetX_Axis;
    { ** Y axis configuration ** }
    property Y_Axis : TAxis read FY_Axis write SetY_Axis;

    { *** Paint procedure (called when ever the graph needs redrawn) *** }
    procedure Paint; override;
  public
    constructor Create(AOwner : TComponent); override;
    destructor Destroy; override;
  end;

implementation

{ ******************************************************** }
{ ************* TAxis object Implementations ************* }
{ ******************************************************** }

constructor TAxis.Create;
begin
  FDivisions := 5;                                                           { Set default values }
  FGridlines := False;
  FMaxValue := 100;
  FMinValue := 0;
  FValues := 3;
  SetRealMaxValue;
end;

procedure TAxis.SetDivisions(Value : Byte);
begin
  If (FDivisions <> Value) And (Value > 1) And (Value >= FValues) then
  begin
    FDivisions := Value;                                                 { Set divisions property }
    SetRealMaxValue;
    If Assigned( FOnValueChange ) then
      OnValueChange( Self );
  end;
end;

procedure TAxis.SetGridlines(Value : Boolean);
begin
  If FGridlines <> Value then
  begin
    FGridlines := Value;                                                 { Set gridlines property }
    If Assigned( FOnValueChange ) then
      OnValueChange( Self );
  end;
end;

procedure TAxis.SetMaxValue(Value : Double);
begin
  If (FMaxValue <> Value) And (Value > FMinValue) Then
  begin
    FMaxValue := Value;                                                  { Set max value property }
    SetRealMaxValue;                                                { Set real max value property }
    If Assigned( FOnValueChange ) then
      OnValueChange( Self );
  end;
end;

procedure TAxis.SetMinValue(Value : Double);
begin
  If (FMinValue <> Value) And (Value < FMaxValue) Then
  begin
    FMinValue := Value;                                                  { Set min value property }
    SetRealMaxValue;                                                { Set real max value property }
    If Assigned( FOnValueChange ) then
      OnValueChange( Self );
  end;
end;

procedure TAxis.SetTitle(Value : String);
begin
  If FTitle <> Value Then
  begin
    FTitle := Value;                                                         { Set title property }
    If Assigned( FOnValueChange ) then                              
      OnValueChange( Self );
  end;
end;

procedure TAxis.SetValues(Value : Byte);
begin
  If (FValues <> Value) And (Value <= FDivisions) Then
  begin
    FValues := Value;                                                       { Set values property }
    SetRealMaxValue;                                                { Set real max value property }
    If Assigned( FOnValueChange ) then
      OnValueChange( Self );
  end;
end;

{ **** Although the max value property specifies the last value on the axis **** }
{ **** it does not mean that the far column of the graph is equal to this   **** }
{ **** value, real max value rectifies this problem                         **** }
procedure TAxis.SetRealMaxValue;
var
  SingleStep : Double;

begin
  If AreDivisions = False then                 { Does the last value fall on the last gridline... }
  begin                                                                                { ...No... }
    If (Spacing > 1) And ((FValues - 1) > 0) then            { ...Calculate how far from the edge }
      SingleStep := (FMaxValue / (FValues - 1)) / Spacing
    else
      SingleStep := 0;
    If (FValues - 1) > 0 then                                      { Calculate the true max value }
      FRealMaxValue := (SingleStep * ((FDivisions - 1) Mod (FValues - 1))) + FMaxValue
    else
      FRealMaxValue := FMaxValue;
  end
  else                                                                                { ...Yes... }
    FRealMaxValue := FMaxValue;
end;

{ ********** Returns True if the last value falls on the last division *********** }
function TAxis.AreDivisions : Boolean;
var
  Ret, ModRet : Integer;

begin
  Ret := (FDivisions + (Spacing - 1)) Div Spacing;
  ModRet := (FDivisions + (Spacing - 1)) Mod Spacing;
  If (Ret = FValues) and (ModRet = 0) then
    Result := True
  else
    Result := False;
end;

{ **** Calculates the number of divisions between each value **** }
function TAxis.Spacing : Byte;
begin
  Result := 1;
  If (FValues - 1) > 0 then
    Result := Abs((FDivisions - 1) Div (FValues - 1));
end;

{ ******************************************************** }
{ ************* TCustomGraph implementations ************* }
{ ******************************************************** }

constructor TCustomGraph.Create(AOwner : TComponent);
begin
  Inherited Create( AOwner );
  FY_Axis := TAxis.Create;                                                 { Create TAxis objects }
  FX_Axis := TAxis.Create;
  FY_Axis.OnValueChange := AxisValueChange;   { Setup procedure when the axis objects are altered }
  FX_Axis.OnValueChange := AxisValueChange;
  Height := 150;                                                                   { Nominal size }
  Width := 200;
  RePaintGraph := True;                                               { Allow graph to be redrawn }
end;

destructor TCustomGraph.Destroy;
begin
  FX_Axis.Destroy;                                                         { Destroy Axis objects }
  FY_Axis.Destroy;
  Inherited Destroy;
end;

{ **** Set property procedures **** }
procedure TCustomGraph.SetAxisColor(Value : TColor);
begin
  If FAxisColor <> Value Then
  begin
    FAxisColor := Value;
    RePaint;
  end;
end;

procedure TCustomGraph.SetBorderStyle(Value : TBorderStyle);
begin
  If FBorderStyle <> Value Then
  begin
    FBorderStyle := Value;
    RePaint;
  end;
end;

procedure TCustomGraph.SetPlotAreaBorderStyle(Value : TBorderStyle);
begin
  If FPlotAreaBorderStyle <> Value then
  begin
    FPlotAreaBorderStyle := Value;
    RePaint;
  end;
end;

procedure TCustomGraph.SetPrintLeft(Value : LongInt);
begin
  if FPrintLeft <> Value Then
    FPrintLeft := Value;
end;

procedure TCustomGraph.SetPrintTop(Value : LongInt);
begin
  if FPrintTop <> Value Then
    FPrintTop := Value;
end;

procedure TCustomGraph.SetPrintHeight(Value : LongInt);
begin
  if (FPrintHeight <> Value) And (Value > 1) Then
    FPrintHeight := Value;
end;

procedure TCustomGraph.SetPrintWidth(Value : LongInt);
begin
  if (FPrintWidth <> Value) And (Value > 1) Then
    FPrintWidth := Value;
end;

procedure TCustomGraph.SetX_Axis(Value : TAxis);
begin
  If FX_Axis <> Value Then
  begin
    FX_Axis := Value;
    RePaint;
  end;
end;

procedure TCustomGraph.SetY_Axis(Value : TAxis);
begin
  If FY_Axis <> Value Then
  begin
    FY_Axis := Value;
    RePaint;
  end;
end;

procedure TCustomGraph.AxisValueChange(Sender : TObject);
begin
  RePaint;
end;

{ ******************** Painting Procedures - TCustomGraph ******************** }
{ **** Draw a vertical line at X,Y for 'Distance' pixels **** }
procedure TCustomGraph.VertLine(var AnImage : TBitmap; X, Y, Distance : Integer);
begin
  With AnImage.Canvas Do
  begin
    MoveTo(X, Y);
    LineTo(X, Y + Distance - 1);
  end;
end;

{ **** Draw a horizontal line at X,Y for 'Distance' pixels **** }
procedure TCustomGraph.HorizLine(var AnImage : TBitmap; X, Y, Distance : Integer);
begin
  With AnImage.Canvas Do
  begin
    MoveTo(X, Y);
    LineTo(X + Distance - 1, Y);
  end;
end;

{ **** Initializes an image by setting the bitmaps dimensions and clearing the background **** }
procedure TCustomGraph.InitImage(var AnImage : TBitmap; BWidth, BHeight : Integer; Col : TColor);
begin
  AnImage.Width := BWidth;
  AnImage.Height := BHeight;
  PaintBackground(AnImage, Col);
end;

{ **** Copies a whole canvas to DestR on the DestC canvas **** }
procedure TCustomGraph.ImageCopy(DestC : TCanvas; DestR : TRect; SourceC : TCanvas);
var
  TR,TempRect : TRect;

begin
  TempRect := SourceC.ClipRect;
  DestC.CopyMode := cmSrcCopy;
  DestC.CopyRect(DestR, SourceC, TempRect);
end;

{ **** Initializes the components drawing area **** }
procedure TCustomGraph.InitDrawGraph(var AnImage : TBitmap; var PRect : TRect);
begin
  PRect := ClientRect;
  InitImage(AnImage, (PRect.Right - PRect.Left), (PRect.Bottom - PRect.Top), clBlack);
  If FBorderStyle = bsSingle then        { Reduces the drawing area if a border needs to be drawn }
    InflateRect(PRect, -1, -1);
  With AnImage.Canvas Do
  begin
    Brush.Color := Color;
    Brush.Style := bsSolid;
    FillRect( PRect );                            { Clears the background with 'Color' prop value }
  end;
end;

{ **** These routines get the rectangular areas of the graph **** }
{ **** The plot area, X axis area and the Y axis area        **** }
{ **** Overlap is where the X axis and the Y axis overlap    **** }
procedure TCustomGraph.GetRects(Sizer : TCanvas;SourceRect : TRect;
                                var YRect, XRect, Overlap : TRect);
var
  FSize, FHeight : Integer;
  XAxisExtra,
  XAxisOffset,
  XMaxHeight,
  YAxisExtra,
  YAxisOffset,
  YMaxWidth      : Word;

begin
  FSize := Abs( Font.Size );                              { Get the current fonts size and height }
  FHeight := Abs( Font.Height );
  { ** This routine returns how many pixels passed the plot area ** }
  { ** the text will span                                        ** }
  GetTextDimensions(Sizer, XAxisExtra, XAxisOffset, YAxisExtra,
                    YAxisOffset, XMaxHeight, YMaxWidth);
  { ** Gets the size of the main plot area ** }
  GetGraphRect(SourceRect, XAxisExtra, YAxisExtra, XMaxHeight, YMaxWidth);
  XRect := GetXAxisRect(SourceRect, XAxisOffset);              { Gets the size of the X axis area }
  YRect := GetYAxisRect(SourceRect, YAxisOffset);              { Gets the size of the Y axis area }
  IntersectRect(Overlap, YRect, XRect);            { Returns the overlapped region of X an Y axis }
end;

{ **** Calculates the extra space needed to centralize the text on each of the divisions **** }
procedure TCustomGraph.GetTextDimensions(Sizer : TCanvas; var XExtra, XOffset, YExtra,
                                         YOffset, XMax, YMax : Word);
begin
  XExtra := 0;
  XOffset := 0;
  YExtra := 0;
  YOffset := 0;
  XMax := GetXText(Sizer, XExtra, XOffset);                    { Gets offsets for the X axis text }
  YMax := GetYText(Sizer, YExtra, YOffset);                    { Gets offsets for the Y axis text }
  If YOffset > XMax then
    XMax := YOffset;
  If XOffset > YMax then
    YMax := XOffset;
end;

{ **** Calculates the extra space needed for the X axis **** }
function TCustomGraph.GetXText(Sizer : TCanvas; var XExtra, XOffset : Word) : Word;
var
  TempSize, FSize, FHeight : Integer;

begin
  Sizer.Font.Assign( Font );
  with Sizer Do
  begin
    FSize := Abs( Font.Size );                                         { Get font size and height }
    FHeight := Abs( Font.Height );
    If FX_Axis.Values > 0 then
    begin
      XOffset := Round(TextWidth( FloatToStr( FX_Axis.MinValue ) ) / 2) + 1;         { Get offset }
      If FX_Axis.AreDivisions = True Then       { If the last value falls on the last division... }
        XExtra := Round(TextWidth( FloatToStr( FX_Axis.MaxValue ) ) / 2) + 1   { ...get the extra }
      Else                                                                           { ...else... }
        XExtra := 4;                                                      { ...set standard extra }
    end
    else
      XExtra := 4;                                                        { ...set standard extra }
  end;
  If FX_Axis.Values <> 0 then                                         { Get height of X axis area }
    Result := Sizer.TextHeight( FloatToStr( X_Axis.MinValue ) ) + 6            { Height of values }
  else
    Result := 6;
  If FX_Axis.Title <> '' then
    Inc(Result, Sizer.TextHeight( X_Axis.Title ) + 6);                          { Height of title }
end;

{ **** Calculates the extra space needed for the Y axis **** }
function TCustomGraph.GetYText(Sizer : TCanvas; var YExtra, YOffset : Word) : Word;
var
  TempStr              : String;
  Step, Current        : Single;
  Biggest              : Word;
  FSize, FHeight, Loop : Integer;

begin
  Sizer.Font.Assign( Font );
  with Sizer Do
  begin
    FSize := Abs( Font.Size );                                         { Get font height and size }
    FHeight := Abs( Font.Height );
    If FY_Axis.Values > 0 then
    begin
      YOffset :=  Round(TextHeight( FloatToStr( Y_Axis.MaxValue) ) / 2) + 1;         { Get offset }
      If FY_Axis.AreDivisions = True then
        YExtra :=  Round(TextHeight( FloatToStr( Y_Axis.MinValue) ) / 2) + 1          { Get extra }
      else
        YExtra := 4;
    end
    else
      YExtra := 4;

    If FY_Axis.Values <> 0 then
    begin
      with FY_Axis Do                               { Get the maximum width needed for the values }
      begin
        If (Values - 1) > 0 then
          Step := (MaxValue - MinValue) / (Values - 1)              { Get the step between values }
        else
          Step := 1;
        Current := MinValue;
        Biggest := 0;
        For Loop := 1 To Values Do                       { Loop that gets the widest value string }
        begin
          TempStr := FloatToStr( Current );
          If TextWidth( TempStr ) > Biggest then
            Biggest := TextWidth( TempStr );
          Current := Current + Step;
        end;
      end;
      Result := Biggest + 6
    end
    Else
      Result := 6;
    If FY_Axis.Title <> '' then                                      { If the axis has a title... }
      Inc(Result, Sizer.TextHeight( FY_Axis.Title ) + 6);       { ...widen the axis for the title }
  end;
end;

{ **** Get the rectangular area that the plotting area takes up **** }
procedure TCustomGraph.GetGraphRect(Source : TRect; XExtra, YExtra, XMax, YMax : Word);
begin
  GraphRect := Source;                             { Source - is the client rect of the component }
  Inc(GraphRect.Left, YMax);                       { Reduce the plot area to accommodate the axii }
  Inc(GraphRect.Top, YExtra);
  Dec(GraphRect.Right, XExtra);
  Dec(GraphRect.Bottom, XMax);
end;

{ **** Get the Y axis rectangular area **** }
function TCustomGraph.GetYAxisRect(Source : TRect; YOffset : Word) : TRect;
begin
  Result := GraphRect;                             { Source - is the client rect of the component }
  Result.Left := Source.Left;
  Result.Top := Source.Top;
  Result.Right := GraphRect.Left;
  Result.Bottom := GraphRect.Bottom + YOffset;
end;

{ **** Get the X axis rectangular area **** }
function TCustomGraph.GetXAxisRect(Source : TRect; XOffset : Word) : TRect;
begin
  Result := GraphRect;
  Result.Left := GraphRect.Left - XOffset;
  Result.Top := GraphRect.Bottom;
  Result.Right := Source.Right;                    { Source - is the client rect of the component }
  Result.Bottom := Source.Bottom;
end;

{ **** Draws the X axis, divisions, titles, values **** }
procedure TCustomGraph.PaintXAxis(var AnImage : TBitmap; XRect : TRect);
var
  XImage : TBitmap;

begin
  XImage := TBitmap.Create;
  try
    { ** Initial the image by painting it the background color ** }
    InitImage(XImage, (XRect.Right - XRect.Left), (XRect.Bottom - XRect.Top), Color);
    XImage.Canvas.Font.Assign( Font );                                  { Assign the correct font }

    If FX_Axis.Values > 0 then
      PaintXAxisValues(XImage, XRect);                               { Paint the values if needed }
    PaintXAxisDivisions(XImage, XRect);                                     { Paint the divisions }
    If FX_Axis.Title <> '' then
      PaintXAxisTitle(XImage, XRect);                                 { Paint the title if needed }

    ImageCopy(AnImage.Canvas, XRect, XImage.Canvas);          { Copy the image to the main bitmap }
  finally
    XImage.Destroy;
  end;
end;

{ **** Paint the X axis values **** }
procedure TCustomGraph.PaintXAxisValues(var AnImage : TBitmap; XRect : TRect);
var
  TempStr                 : String;
  TextStep, CentreX,
  TextValue, Step, Actual : Single;
  FHeight, FSize,
  Loop, Current           : Integer;

begin
  FSize := Abs( Font.Size );                                           { Get font size and height }
  FHeight := Abs( Font.Height );
  with FX_Axis Do                                                   { Get spacings between values }
  begin                                                    { Get incrementations between spacings }
    If (Values - 1) > 0 then
      TextStep := (MaxValue  - MinValue) / (Values - 1)
    else
      TextStep := 1;
    Step := ((GraphRect.Right - GraphRect.Left) / (Divisions - 1)) * Spacing;
    TextValue := MinValue;
  end;
  Actual := GraphRect.Left - XRect.Left;
  with AnImage.Canvas Do
  begin
    Brush.Style := bsClear;
    For Loop := 1 To FX_Axis.Values Do
    begin
      TempStr := FloatToStr( TextValue );
      CentreX := TextWidth( TempStr ) / 2;
      Current := Round(Actual - CentreX);                        { Calculate the correct position }

      TextOut(Current, 7 - (FHeight - FSize), TempStr);             { Draw the text to the bitmap }

      Actual := Actual + Step;                                    { Increment the positions, etc. }
      TextValue := TextValue + TextStep;
    end;
  end;
end;

{ **** Paint the X axis title **** }
procedure TCustomGraph.PaintXAxisTitle(var AnImage : TBitmap; XRect : TRect);
var
  FSize, FHeight, XTop, GWidth, CentreX : Integer;

begin
  GWidth := GraphRect.Right - GraphRect.Left;              { Calculate the width of the plot area }
  With AnImage.Canvas Do
  begin
    FHeight := Abs( Font.Height );
    FSize := Abs( Font.Size );
    CentreX := Round((GWidth - TextWidth( FX_Axis.Title )) / 2);  { Get the x co-ord of the title }
    Inc(CentreX, (GraphRect.Left - XRect.Left));               { Centre the text to the plot area }
    XTop := (XRect.Bottom - (TextHeight( FX_Axis.Title ) + 6)) - XRect.Top;
    TextOut(CentreX, XTop, FX_Axis.Title);                        { Output the text to the bitmap }
  end;
end;

{ **** Paint the X axis divisions **** }
procedure TCustomGraph.PaintXAxisDivisions(var AnImage : TBitmap; XRect : TRect);
var
  Step, Actual  : Single;
  Loop, Current : Integer;

begin
  Step := (GraphRect.Right - GraphRect.Left) / (FX_Axis.Divisions - 1);                { Get step }
  Actual := GraphRect.Left - XRect.Left;                              { Get actual starting point }
  with AnImage.Canvas Do
  begin
    Pen.Color := FAxisColor;
    Pen.Mode := pmCopy;
    Pen.Style := psSolid;
    Pen.Width := 1;
  end;
  For Loop := 1 To FX_Axis.Divisions Do                     { Paint the divisions onto the bitmap }
  begin
    Current := Round( Actual );
    VertLine(AnImage, Current, 0, 4);
    Actual := Actual + Step;
  end;
end;

{ **** Draws the Y axis, values, divisions, title **** }
procedure TCustomGraph.PaintYAxis(var AnImage : TBitmap; YRect, Overlap : TRect);
var
  TempRect : TRect;
  YImage   : TBitmap;

begin
  YImage := TBitmap.Create;
  try
    { ** Initialize the image by clearing it to the background color ** }
    InitImage(YImage, (YRect.Right - YRect.Left), (YRect.Bottom - YRect.Top), Color);
    YImage.Canvas.Font.Assign( Font );

    with YImage.Canvas Do
    begin
      TempRect := Overlap;
      If FBorderStyle = bsSingle then
        OffsetRect(TempRect, -1, -1);
      CopyMode := cmSrcCopy;
      { ** Copy the overlapped image to the y axis bitmap ** }
      CopyRect(TempRect, AnImage.Canvas, Overlap);
    end;

    If FY_Axis.Values > 0 then
      PaintYAxisValues(YImage, YRect);                               { Paint the values if needed }
    PaintYAxisDivisions(YImage, YRect);                                     { Paint the divisions }
    If FY_Axis.Title <> '' then
      PaintYAxisTitle(YImage, YRect);                                 { Paint the title if needed }

    ImageCopy(AnImage.Canvas, YRect, YImage.Canvas);          { Copy the image to the main bitmap }
  finally
    YImage.Destroy;
  end;
end;

{ **** Paint the Y axis title **** }
procedure TCustomGraph.PaintYAxisTitle(var AnImage : TBitmap; YRect : TRect);
var
  FSize, FHeight, CentreY : Integer;

begin
  AnImage.Canvas.Font.Assign( Font );
  with AnImage.Canvas Do
  begin
    FHeight := Abs( Font.Height );
    FSize := Abs( Font.Size );
    { ** Calculate the central position of the title ** }
    CentreY := Round((((GraphRect.Bottom - GraphRect.Top) / 2) + (GraphRect.Top - YRect.Top)) +
               (TextWidth( FY_Axis.Title ) / 2));
    { ** Draw the text at 90 to normal ** }
    AngleTextOut(AnImage.Canvas, 900, 2, CentreY, FY_Axis.Title);
  end;
end;

{ **** Paint the Y axis values **** }
procedure TCustomGraph.PaintYAxisValues(var AnImage : TBitmap; YRect : TRect);
var
  TempStr                    : String;
  TextValue, Step, Actual,
  TextStep, CentreY, CentreX : Double;
  FSize,
  Loop, Current, FHeight     : Integer;

begin
  FSize := Abs( Font.Size );
  FHeight := Abs( Font.Height );
  with FY_Axis Do
  begin
    If (Values - 1) > 0 then                                                 { Get the value step }
      TextStep := (MaxValue  - MinValue) / (Values - 1)
    else
      TextStep := 1;
    { ** Get the pixel step ** }
    Step := ((GraphRect.Bottom - GraphRect.Top) / (Divisions - 1)) * Spacing;
    TextValue := MinValue;
  end;
  Actual := GraphRect.Bottom - YRect.Top;                               { Get the actual position }
  with AnImage.Canvas Do
  begin
    Brush.Style := bsClear;
    For Loop := 1 To FY_Axis.Values Do
    begin
      TempStr := FloatToStr( TextValue );                         { Convert the value to a string }
      CentreY := TextHeight( TempStr ) / 2;
      Current := (Round( Actual ) - Round( CentreY )) - ((FHeight - FSize) Div 2);
      CentreX := TextWidth( TempStr ) + 6;
      CentreX := (YRect.Right - YRect.Left + 1) - CentreX;

      TextOut(Round( CentreX ), Current, TempStr);                               { Paint the text }

      Actual := Actual - Step;                                           { Increment the position }
      TextValue := TextValue + TextStep;
    end;
  end;
end;

{ **** Draw the Y axis divisions **** }
procedure TCustomGraph.PaintYAxisDivisions(var AnImage : TBitmap; YRect : TRect);
var
  Step, Actual  : Single;
  Loop, Current : Integer;

begin
  Step := (GraphRect.Bottom - GraphRect.Top) / (FY_Axis.Divisions - 1);         { Get pixels step }
  Actual := GraphRect.Bottom - YRect.Top - 1;                                { Get starting point }
  with AnImage.Canvas Do
  begin
    Pen.Color := FAxisColor;
    Pen.Mode := pmCopy;
    Pen.Style := psSolid;
    Pen.Width := 1;
  end;
  For Loop := 1 To FY_Axis.Divisions Do
  begin
    Current := Round( Actual );                                            { Get rounded position }
    HorizLine(AnImage, YRect.Right - YRect.Left, Current, -4);                   { Draw divisions }
    Actual := Actual - Step;                                                 { Increment position }
  end;
end;

{ **** Paint the X and Y axis gridlines **** }
procedure TCustomGraph.PaintGridlines(var AnImage : TBitmap);
begin
  PaintGraphBorder( AnImage );                           { Paints the plot areas border if needed }
  If X_Axis.Gridlines = True then
    PaintXGridlines( AnImage );                            { Paint the X axis gridlines if needed }
  If Y_Axis.Gridlines = True then
    PaintYGridlines( AnImage );                            { Paint the Y axis gridlines if needed }
end;

{ **** Draws the plot areas border if needed **** }
procedure TCustomGraph.PaintGraphBorder(var AnImage : TBitmap);
var
  GraphWidth, GraphHeight : Integer;

begin
  GraphWidth := GraphRect.Right - GraphRect.Left;
  GraphHeight := GraphRect.Bottom - GraphRect.Top;
  With AnImage.Canvas Do                                            { Setup the brush and the pen }
  begin
    With Brush Do
    begin
      Color := AxisColor;
      Style := bsSolid;
    end;
    With Pen Do
    begin
      Color := AxisColor;
      Style := psSolid;
      Mode := pmCopy;
      Width := 1;
    end;
    If PlotAreaBorderStyle = bsSingle then                              { If there is a border... }
      FrameRect( Rect(0, 0, GraphWidth, GraphHeight) )                               { ...draw it }
    else                                                                        { ...otherwise... }
    begin
      MoveTo(0, 0);                                               { ...just draw the X and Y axis }
      LineTo(0, GraphHeight - 1);
      LineTo(GraphWidth, GraphHeight - 1);
    end;
  end;
end;

{ **** Paint the X axis gridlines **** }
procedure TCustomGraph.PaintXGridlines(var AnImage : TBitmap);
var
  Step, Actual               : Single;
  GraphHeight, Loop, Current : Integer;

begin
  { ** Get the pixels step between gridlines ** }
  Step := ((GraphRect.Right - GraphRect.Left) - 1) / (X_Axis.Divisions - 1);
  GraphHeight := GraphRect.Bottom - GraphRect.Top;
  Actual := 0;
  with AnImage.Canvas Do
  begin
    Pen.Color := AxisColor;
    Pen.Mode := pmCopy;
    Pen.Style := psSolid;
    Pen.Width := 1;
  end;
  For Loop := 1 To X_Axis.Divisions - 1 Do
  begin
    Actual := Actual + Step;                                                 { Increment position }
    Current := Round( Actual );                                        { Get the current position }
    VertLine(AnImage, Current, 0, GraphHeight);                               { Draw the gridline }
  end;
end;

{ **** Paint the Y axis gridlines **** }
procedure TCustomGraph.PaintYGridlines(var AnImage : TBitmap);
var
  Step, Actual              : Single;
  GraphWidth, Loop, Current : Integer;

begin
  { ** Get the pixels step between gridlines ** }
  Step := ((GraphRect.Bottom - GraphRect.Top) - 1) / (Y_Axis.Divisions - 1);
  GraphWidth := GraphRect.Right - GraphRect.Left;
  Actual := 0;
  with AnImage.Canvas Do
  begin
    Pen.Color := AxisColor;
    Pen.Mode := pmCopy;
    Pen.Style := psSolid;
    Pen.Width := 1;
  end;
  For Loop := 1 To Y_Axis.Divisions - 1 Do
  begin
    Current := Round( Actual );                                        { Get the current position }
    HorizLine(AnImage, 0, Current, GraphWidth);                               { Draw the gridline }
    Actual := Actual + Step;
  end;
end;

{ **** Paints the plot area bit **** }
procedure TCustomGraph.PaintGraph(var AnImage : TBitmap);
var
  TempRect                         : TRect;
  GraphWidth, GraphHeight          : Integer;
  StartX, EndX, StepX,
  HPixelsPerValue, VPixelsPerValue : Single;
  GImage                           : TBitmap;

begin
  GImage := TBitmap.Create;
  try
    Dec( GraphRect.Top );
    Inc( GraphRect.Right );
    GraphWidth := GraphRect.Right - GraphRect.Left;
    GraphHeight := GraphRect.Bottom - GraphRect.Top;
    InitImage(GImage, GraphWidth, GraphHeight, Color);                     { Initialize the image }

    { ** Get the calibration factors ** }
    GetCalibration(VPixelsPerValue, HPixelsPerValue, StartX, EndX, StepX);
    { ** Paint the gridlines (if neccesary) }
    PaintGridlines( GImage );
    { ** Paint the data ** }
    PaintDataLines(GImage, VPixelsPerValue, HPixelsPerValue);

    { ** Copy the image to the main bitmap ** }
    ImageCopy(AnImage.Canvas, GraphRect, GImage.Canvas);
  finally
    GImage.Destroy;
  end;
end;

{ **** Paint a bitmapped image's background 'Col' color **** }
procedure TCustomGraph.PaintBackground(var AnImage : TBitmap; Col : TColor);
begin
  with AnImage.Canvas Do
  begin
    Brush.Color := Col;
    Brush.Style := bsSolid;
    FillRect(Rect(0, 0, AnImage.Width, AnImage.Height));
  end;
end;

{ **** This is the virtual procedure overridden from the TGraphicControl **** }
procedure TCustomGraph.Paint;
var
  GraphImage : TBitmap;
  ORect,
  YAxisRect,
  XAxisRect,
  PaintRect  : TRect;

begin
  If RePaintGraph = True then                                  { If you are allowed to repaint... }
  begin                                                                    { ...Repaint the graph }
    GraphImage := TBitmap.Create;
    try
      InitDrawGraph(GraphImage, PaintRect);                      { Initialize the painting bitmap }

      { ** Get the graphs rectangular regions ** }
      GetRects(GraphImage.Canvas, PaintRect, YAxisRect, XAxisRect, ORect);

      If XAxisRect.Top <> XAxisRect.Bottom Then
        PaintXAxis(GraphImage, XAxisRect);                                     { Paint the X axis }
      If YAxisRect.Top <> YAxisRect.Bottom Then
        PaintYAxis(GraphImage, YAxisRect, ORect);                              { Paint the Y axis }
      If GraphRect.Top <> GraphRect.Bottom then
        PaintGraph( GraphImage );                                           { Paint the plot area }

      ImageCopy(Canvas, ClientRect, GraphImage.Canvas); { Copy the image to the components canvas }
    finally
      GraphImage.Destroy;
    end;
  end;
end;

{ ************* TCustomGraph Printing Procedures ************* }
procedure TCustomGraph.InitPrintImage(var PRect : TRect);
begin
  With Printer.Canvas Do
  begin
    PRect := ClipRect;
    PRect.Right := PrintWidth;
    PRect.Bottom := PrintHeight;
    OffsetRect(PRect, FPrintLeft, FPrintTop);       { Setup where the printed graph should appear }
    Brush.Color := clBlack;
    Brush.Style := bsSolid;
    FillRect( PRect );
    If FBorderStyle = bsSingle Then
      InflateRect(PRect, -3, -3);                                 { Print the border if neccesary }
    Brush.Color := Color;
    FillRect( PRect );                                                     { Print the background }
    Pen.Style := psSolid;
    Pen.Mode := pmCopy;
    Pen.Width := 3;
    Brush.Style := bsClear;
    Font.Assign( Font );
  end;
end;

{ **** Makes up for errors brough about by the printers high resolution **** }
procedure TCustomGraph.SetPrinterRects(var XRect, YRect : TRect);
begin
  If FBorderStyle = bsSingle Then
  begin
    Dec(XRect.Right, 12);
    Dec(GraphRect.Right, 12);
    Inc(YRect.Top, 12);
    Inc(GraphRect.Top, 12);
    OffsetRect(XRect, 0, -12);
    OffsetRect(YRect, 12, 0);
    GraphRect.Bottom := XRect.Top - 1;
    GraphRect.Left := YRect.Right + 1;
  end;
end;

{ **** Prints the whole graph at PrintLeft, PrintTop **** }
{ ****             with size PrintWidth, PrintHeight **** }
procedure TCustomGraph.PrintGraph;
var
  ORect,
  XAxisRect,
  YAxisRect,
  PrintRect : TRect;

begin
  InitPrintImage( PrintRect );                                     { Initialize the printed image }
  GetRects(Printer.Canvas, PrintRect, YAxisRect, XAxisRect, ORect);    { Get the graph rectangles }
  SetPrinterRects(XAxisRect, YAxisRect);                       { Modify the rects for the printer }

  PrintXAxis( XAxisRect );                                                     { Print the X axis }
  PrintYAxis( YAxisRect );                                                     { Print the Y axis }
  PrintGridlines;                                                           { Print the gridlines }
end;

{ **** Prints the X axis **** }
procedure TCustomGraph.PrintXAxis(XRect : TRect);
begin
  If FX_Axis.Values > 0 Then
    PrintXAxisValues( XRect );                                      { Prints the values if needed }
  PrintXAxisDivisions( XRect );                                            { Prints the divisions }
  If FX_Axis.Title <> '' Then
    PrintXAxisTitle( XRect );                                        { Prints the title if needed }
end;

{ **** Prints the X axis values **** }
procedure TCustomGraph.PrintXAxisValues(XRect : TRect);
var
  TempStr                 : String;
  TextStep, CentreX,
  TextValue, Step, Actual : Double;
  FHeight, FSize,
  Loop, Current           : Integer;

begin
  With FX_Axis Do
  begin
    If (Values - 1) > 0 then                                   { Get the step between text values }
      TextStep := (MaxValue - MinValue) / (Values - 1)
    else
      TextStep := 1;
    { ** Get the step between divisions ** }
    Step := ((GraphRect.Right - GraphRect.Left) / (Divisions - 1)) * Spacing;
    TextValue := MinValue;
  end;
  Printer.Canvas.Font.Assign( Font );
  With Printer.Canvas Do
  begin
    FSize := Abs( Font.Size );
    FHeight := Abs( Font.Height );
    Actual := GraphRect.Left;
    For Loop := 1 To FX_Axis.Values Do
    begin
      TempStr := FloatToStr( TextValue );
      CentreX := TextWidth( TempStr ) / 2;                           { Calculate the Texts centre }
      Current := Round( Actual ) - Round( CentreX );

      TextOut(Current, XRect.Top + 16, TempStr);                                 { Print the text }

      Actual := Actual + Step;
      TextValue := TextValue + TextStep;
    end;
  end;
end;

{ **** Prints the X axis divisions **** }
procedure TCustomGraph.PrintXAxisDivisions(XRect : TRect);
var
  Step, Actual  : Single;
  Loop, Current : Integer;

begin
  { ** Get the number of pixels between divisions ** }
  Step := (GraphRect.Right - GraphRect.Left) / (FX_Axis.Divisions - 1);
  Actual := GraphRect.Left;
  With Printer.Canvas Do
  begin
    Pen.Color := FAxisColor;
    For Loop := 1 To FX_Axis.Divisions Do
    begin
      Current := Round( Actual );                                          { Get rounded position }

      MoveTo(Current, XRect.Top);
      LineTo(Current, XRect.Top + 16);                                            { Draw division }

      Actual := Actual + Step;                                               { Increment position }
    end;
  end;
end;

{ **** Print the X axis title **** }
procedure TCustomGraph.PrintXAxisTitle(XRect : TRect);
var
  FSize, FHeight, XTop, GWidth, CentreX : Integer;

begin
  GWidth := GraphRect.Right - GraphRect.Left;
  Printer.Canvas.Font.Assign( Font );
  With Printer.Canvas Do
  begin
    FHeight := Abs( Font.Height );
    FSize := Abs( Font.Size );
    CentreX := Round((GWidth - TextWidth( FX_Axis.Title )) / 2);            { Get centre of title }
    Inc(CentreX, GraphRect.Left);                       { Offset the title to centre on the graph }
    XTop := XRect.Bottom - (FHeight + 20);                       { Get the Y co-ord for the title }
    TextOut(CentreX, XTop, FX_Axis.Title);                                       { Print the text }
  end;
end;

{ **** Prints all components of the Y axis **** }
procedure TCustomGraph.PrintYAxis(YRect : TRect);
begin
  If FY_Axis.Values > 0 Then
    PrintYAxisValues( YRect );                                { Print the Y axis values if needed }
  PrintYAxisDivisions( YRect );                                             { Print the divisions }
  If FY_Axis.Title <> '' Then
    PrintYAxisTitle( YRect );                                  { Print the Y axis title if needed }
end;

{ **** Print the Y axis values **** }
procedure TCustomGraph.PrintYAxisValues(YRect : TRect);
var
  TempStr                       : String;
  TextValue, Step, Actual,
  TextStep, CentreY, CentreX    : Double;
  Loop, Current, FHeight, FSize : Integer;

begin
  With FY_Axis Do
  begin
    If (Values - 1) > 0 then                                   { Get the step between text values }
      TextStep := (MaxValue - MinValue) / (Values - 1)
    else
      TextStep := 1;
    { ** Get the number of pixels between value centers ** }
    Step := ((GraphRect.Bottom - GraphRect.Top) / (Divisions - 1)) * Spacing;
    TextValue := MinValue;
  end;
  Actual := GraphRect.Bottom;
  Printer.Canvas.Font.Assign( Font );
  With Printer.Canvas Do
  begin
    FHeight := Abs( Font.Height );
    FSize := Abs( Font.Size );
    For Loop := 1 To FY_Axis.Values Do
    begin
      TempStr := FloatTostr( TextValue );                             { Convert value to a string }
      CentreY := TextHeight( TempStr ) / 2;                                  { Get height of text }
      Current := Round(Actual - CentreY);
      CentreX := TextWidth( TempStr ) + 24;
      CentreX := (YRect.Right + 1) - CentreX;              { Get the central location of the text }

      TextOut(Round( CentreX ), Current, TempStr);                              { Print the value }

      Actual := Actual - Step;                                           { Increment the position }
      TextValue := TextValue + TextStep;                            { Increment to the next value }
    end;
  end;
end;

{ **** Print the Y axis divisions **** }
procedure TCustomGraph.PrintYAxisDivisions(YRect : TRect);
var
  Step, Actual  : Single;
  Loop, Current : Integer;

begin
  { ** Get the number of pixels between divisions ** }
  Step := (GraphRect.Bottom - GraphRect.Top) / (FY_Axis.Divisions - 1);
  Actual := GraphRect.Bottom;
  with Printer.Canvas Do
  begin
    Pen.Color := FAxisColor;
    For Loop := 1 To FY_Axis.Divisions Do
    begin
      Current := Round( Actual );                                      { Get the rounded position }

      MoveTo(YRect.Right, Current);
      LineTo(YRect.Right - 16, Current);                                     { Print the division }

      Actual := Actual - Step;                                           { Increment the position }
    end;
  end;
end;

{ **** Prints the Y axis title **** }
procedure TCustomGraph.PrintYAxisTitle(YRect : TRect);
var
  FSize, FHeight, CentreY : Integer;

begin
  Printer.Canvas.Font.Assign( Font );
  With Printer.Canvas Do
  begin
    FHeight := Abs( Font.Height );
    FSize := Abs( Font.Size );
    { ** Get the central location for the title ** }
    CentreY := Round(((GraphRect.Bottom - GraphRect.Top) / 2) + GraphRect.Top);
    CentreY := CentreY + Round(TextWidth( FY_Axis.Title ) / 2);
  end;
  { ** Print the text at 90 clock-wise to normal ** }
  AngleTextOut(Printer.Canvas, 900, YRect.Left, CentreY, FY_Axis.Title);
end;

{ ** Print the plot areas gridlines ** }
procedure TCustomGraph.PrintGridlines;
begin
  PrintGraphBorder;                                                 { Print the plot areas border }
  If FX_Axis.Gridlines = True then
    PrintXGridlines;                                       { Print the X axis gridlines if needed }
  If FY_Axis.Gridlines = True then
    PrintYGridlines;                                       { Print the Y axis gridlines if needed }
end;

{ **** Prints the plot areas border **** }
procedure TCustomGraph.PrintGraphBorder;
begin
  With Printer.Canvas Do
  begin
    with Pen Do
      Color := FAxisColor;
    MoveTo(GraphRect.Left, GraphRect.Top);
    LineTo(GraphRect.Left, GraphRect.Bottom);
    LineTo(GraphRect.Right, GraphRect.Bottom);                     { Prints the X and Y axis bars }
    If FPlotAreaBorderStyle = bsSingle then                             { If there is a border... }
    begin
      LineTo(GraphRect.Right, GraphRect.Top);                                { ...draw the border }
      LineTo(GraphRect.Left, GraphRect.Top);
    end;
  end;
end;

{ **** Print the X axis gridlines (verticals) **** }
procedure TCustomGraph.PrintXGridlines;
var
  Step, Actual  : Single;
  Loop, Current : Integer;

begin
  { ** Distance (in pixels) between gridlines ** }
  Step := (GraphRect.Right - GraphRect.Left) / (FX_Axis.Divisions - 1);
  Actual := GraphRect.Left;
  With Printer.Canvas Do
  begin
    Pen.Color := FAxisColor;
    Pen.Width := 1;
    For Loop := 1 TO FX_Axis.Divisions - 2 Do
    begin
      Actual := Actual + Step;                                       { Increment current position }
      Current := Round( Actual );

      MoveTo(Current, GraphRect.Bottom);                                     { Print the gridline }
      LineTo(Current, GraphRect.Top);
    end;
    Pen.Width := 3;
    Actual := Actual + Step;
    Current := Round( Actual );

    MoveTo(Current, GraphRect.Bottom);
    LineTo(Current, GraphRect.Top);                     { Print the last gridline (it is thicker) }
  end;
end;

{ **** Print the Y axis gridlines (horizontals) **** }
procedure TCustomGraph.PrintYGridlines;
var
  Step, Actual  : Single;
  Loop, Current : Integer;

begin
  { Distance (in pixels) between the gridlines ** }
  Step := (GraphRect.Bottom - GraphRect.Top) / (FY_Axis.Divisions - 1);
  Actual := GraphRect.Top;
  With Printer.Canvas Do
  begin
    Pen.Color := FAxisColor;
    Pen.Width := 3;
    Current := Round( Actual );

    MoveTo(GraphRect.Left, Current);
    LineTo(GraphRect.Right, Current);                  { Print the first gridline (it is thicker) }

    Actual := Actual + Step;
    Pen.Width := 1;
    For Loop := 1 To FY_Axis.Divisions - 2 Do
    begin
      Current := Round( Actual );

      MoveTo(GraphRect.Left, Current);
      LineTo(GraphRect.Right, Current);                                      { Print the gridline }

      Actual := Actual + Step;                                           { Increment the position }
    end;
    Pen.Width := 3;
  end;
end;

{ **************************************************************** }
{ ************* TDoubleStream object Implementations ************* }
{ **************************************************************** }
{ **** This component is basically a memorystream object **** }
{ **** but it orders it's data into 8 byte chunks which  **** }
{ **** make up a Double variable                         **** }
constructor TDoubleStream.Create;
begin
  Data := TMemoryStream.Create;                                        { Create the TMemoryStream }
  Data.Clear;
end;

destructor TDoubleStream.Free;
begin
  Data.Free;                                                           { Delete the TMemoryStream }
end;

{ **** Read function for the Size property **** }
{ **** Divides the Memorystreams size by 8 **** }
function TDoubleStream.GetSize : LongInt;
begin
  Result := Data.Size Div 8;
end;

{ **** Reads a value from the stream **** }
function TDoubleStream.Read(Idx : LongInt) : Double;
var
  Temp : LongInt;

begin
  If (Idx < 1) Or (Idx > Data.Size) then
    raise EListError.Create( 'Stream Index Out of range' );         { Raise error if out of range }
  Data.Seek((Idx - 1) * 8, Seek_Begin);                    { Seek the value within the data array }
  Temp := Data.Read(Result, 8);                                  { Read the value from the stream }
end;

{ **** Writes over an existing value **** }
procedure TDoubleStream.Write(Idx : LongInt; Value : Double);
var
  Temp : LongInt;

begin
  If (Idx < 1) Or (Idx > Data.Size) then
    raise EListError.Create( 'Stream Index Out of range' );         { Raise error if out of range }
  Data.Seek((Idx - 1) * 8, Seek_Begin);                        { Seek the value in the data array }
  Data.Write(Value, 8);                                                    { Write over the value }
end;

{ **** Add a new value to the end of the stream **** }
procedure TDoubleStream.Add(Value : Double);
begin
  Data.Seek(0, Seek_End);                                        { Goto the end of the data array }
  Data.Write(Value, 8);                                                     { Write the new value }
end;

{ **** Clear all existing data out of the stream **** }
procedure TDoubleStream.Clear;
begin
  Data.Clear;
end;

{ **** Save the stream to a file on disk **** }
procedure TDoubleStream.SaveToFile(Filename : String);
begin
  Data.SaveToFile( Filename );
end;

{ **** Load a stream from a file on disk **** }
procedure TDoubleStream.LoadFromFile(Filename : String);
begin
  Data.LoadFromFile( Filename );
end;

end.

